precision mediump float;

// special hack for the red car model: the red parts are metallic so they should be shinier
// when not defined: use the alpha channel for specular power: exponent = 1.0 + tex.a*2048.0 (tex.a: 0..1)
//#define HACK_USE_RED_CHANNEL_FOR_SPECULARITY

varying vec2 v_txCoor;
varying vec3 v_EyeSpaceNormal;
varying vec3 v_EyeSpacePos;
varying vec3 v_ObjectSpaceNormal;

uniform sampler2D texture;

vec3 vCameraPosition = vec3(0.0);	// eye space position
#define C_PI 3.1415926535897932384626433832795

// "GI": Valve ambient cube, http://www.valvesoftware.com/publications/2006/SIGGRAPH06_Course_ShadingInValvesSourceEngine.pdf
// eye space
// these values will be multiplied by cAmbient
vec3 ambientCubePX = vec3(0.450, 0.130, 0.400);		// viewport right
vec3 ambientCubeNX = vec3(0.150, 0.130, 0.720);		// viewport left
vec3 ambientCubePY = vec3(0.520, 0.450, 0.200);		// viewport top
vec3 ambientCubeNY = vec3(0.100, 0.420, 0.560);		// viewport bottom
vec3 ambientCubePZ = vec3(0.200, 0.200, 0.350);		// near
vec3 ambientCubeNZ = vec3(0.000, 0.330, 0.630);		// far

// material parameters
vec3 cAmbient = vec3(0.6, 0.6, 0.6);
vec3 cDiffuse;
float fSpecularExponent;
float fFresnel;
float fSpecularWeight = 1.0;

// temp
vec3 N;
vec3 V;

vec3 GetAmbientFromCube()
{
	vec3 ambient = vec3(0.0);
	
	//vec3 param = normalize(v_ObjectSpaceNormal);
	vec3 param = reflect(-V, N);
	vec3 paramP = clamp(param, 0.0, 1.0);
	vec3 paramN = clamp(-param, 0.0, 1.0);

	ambient += paramP.x * ambientCubePX;
	ambient += paramN.x * ambientCubeNX;
	ambient += paramP.y * ambientCubePY;
	ambient += paramN.y * ambientCubeNY;
	ambient += paramP.z * ambientCubePZ;
	ambient += paramN.z * ambientCubeNZ;

	return ambient;
}

vec3 CalculateFresnelSpecularDiffuseAmbient(vec3 vLightPos, vec3 cLight)
{
	// energy conservation: the reflected light amount should be less than the incoming light
	// http://www.rorydriscoll.com/2009/01/25/energy-conservation-in-games/
	float specularEnergyConservationMul = (fSpecularExponent + 8.0) / (8.0*C_PI);

	vec3 L = normalize(vLightPos-v_EyeSpacePos);
	vec3 H = normalize(V+L);

	float dotNL = max( dot(N, L), 0.0 );
	float dotNH = max( dot(N, H), 0.0 );
	float dotVH = max( dot(V, H), 0.0 );

	float fresnelBase = 1.0 - dotVH;
	float fresnelExp = pow(fresnelBase, 5.0);
	float fresnel = fresnelExp + fFresnel * (1.0-fresnelExp);

	vec3 spec = vec3(pow(dotNH, fSpecularExponent)) * cDiffuse;
	vec3 diff = vec3(dotNL) * cDiffuse;
	
	// final color, in linear space
	vec3 final = vec3(0.0);
	final += cLight * fSpecularWeight * fresnel * spec * specularEnergyConservationMul * dotNL;
	final += cLight * diff / C_PI;

	return final;
}

void main()
{
	// temp variables for lighting
	N = normalize(v_EyeSpaceNormal);
	V = normalize(vCameraPosition-v_EyeSpacePos);
	
	// get texture color and put it into linear space
	// we assume that the texture is in sRGB
	// http://filmicgames.com/archives/299
	vec4 cTex = texture2D(texture, v_txCoor);
	cDiffuse = pow(cTex.rgb, vec3(2.2));
	
	// specular exponent hack
#ifdef HACK_USE_RED_CHANNEL_FOR_SPECULARITY
	float specularity = cDiffuse.r*cDiffuse.r;
	fSpecularExponent = 20.0 + specularity * 400.0;
	fFresnel = 0.2 + specularity * (0.95-0.2);

#else
	fSpecularExponent = 1.0 + cTex.a*2048.0;
	fFresnel = 0.75;

#endif

	//vec3 keyLight = vec3(0, 10.0, -10.0);
	vec3 keyLight = vec3(0, 10.0, -10.0);
	vec3 bounceLight = vec3(0, -10.0, 10.0);
	vec3 light5 = vec3(0.0, 17.0, 20.0);
	vec3 light10 = vec3(-20.0, -28.0, 0.0);
	vec3 light11 = vec3(20.0, 32.0, 0.0);

	vec3 final = GetAmbientFromCube() * cAmbient * cDiffuse;
	//vec3 final = GetAmbientFromCube();
	final += CalculateFresnelSpecularDiffuseAmbient( keyLight, vec3(1.35, 1.35, 1.35) );		// key light
	final += CalculateFresnelSpecularDiffuseAmbient( bounceLight, vec3(0.45, 0.68, 0.78) );		// bounce
	final += CalculateFresnelSpecularDiffuseAmbient( light5, vec3(0.65, 0.63, 0.60) );			// fill
	final += CalculateFresnelSpecularDiffuseAmbient( light10, vec3(0.65, 0.63, 0.60) );			// fill
	final += CalculateFresnelSpecularDiffuseAmbient( light11, vec3(0.65, 0.63, 0.60) );			// fill
	
	// gamma uncorrect :)
	// http://filmicgames.com/archives/299
	gl_FragColor.rgb = pow( final, vec3(1.0/2.2) );
	gl_FragColor.a = cTex.a;
}
